#!/usr/bin/env python
"""
Build a DR5 random lens catalog that matches the
(Mstar_log10, R_G_kpc, z_lens, bins) distribution
of data/lenses_dr5.csv but with random sky positions.

This is a simple rectangular-footprint random:
ra_deg ~ Uniform(min_ra, max_ra)
dec_deg ~ Uniform(min_dec, max_dec)

If you later want a more precise footprint mask,
you can swap out the RA/Dec generation logic.
"""

import numpy as np
import pandas as pd
import os


def main():
    lenses_path = "data/lenses_dr5.csv"
    out_path = "data/lenses_dr5_random.csv"

    if not os.path.exists(lenses_path):
        raise FileNotFoundError(f"Missing {lenses_path}")

    df = pd.read_csv(lenses_path)

    required_cols = [
        "lens_id",
        "ra_deg",
        "dec_deg",
        "z_lens",
        "R_G_kpc",
        "Mstar_log10",
        "R_G_bin",
        "Mstar_bin",
    ]
    for c in required_cols:
        if c not in df.columns:
            raise ValueError(f"Column '{c}' missing from {lenses_path}")

    # How many randoms? Use a simple factor of 10 × number of lenses.
    n_lens = len(df)
    n_rand = 10 * n_lens

    rng = np.random.default_rng(12345)

    # RA/Dec bounds from the existing DR5 lenses
    ra_min, ra_max = df["ra_deg"].min(), df["ra_deg"].max()
    dec_min, dec_max = df["dec_deg"].min(), df["dec_deg"].max()

    # Sample indices from the lens catalog (with replacement)
    idx = rng.integers(0, n_lens, size=n_rand)
    base = df.iloc[idx].reset_index(drop=True).copy()

    # Assign new sky positions uniformly in the bounding box
    base["ra_deg"] = rng.uniform(ra_min, ra_max, size=n_rand)
    base["dec_deg"] = rng.uniform(dec_min, dec_max, size=n_rand)

    # Give them new IDs so they don't collide with lens_id
    base["lens_id"] = [f"rand_{i}" for i in range(n_rand)]

    # Keep the same physical columns/distributions
    # (z_lens, R_G_kpc, Mstar_log10, R_G_bin, Mstar_bin) already in base.

    base.to_csv(out_path, index=False)
    print(f"Wrote randoms to {out_path} (n={n_rand})")


if __name__ == "__main__":
    main()
